/**
 * Cocos Event Tracker SDK
 *
 * A JavaScript/TypeScript SDK for tracking and sending events to Cocos OLAP service
 * with built-in time synchronization and retry mechanisms.
 *
 * @author Cocos Team
 */
/**
 * Cocos Event Tracker SDK Main Class
 */
class CocosEventTracker {
    constructor(config = {}) {
        var _a;
        this.isInitialized = false;
        this.startupServerTimestamp = null;
        this.clientInitTime = null;
        this.eventQueue = [];
        this.batchTimer = null;
        this.currentUserId = null;
        this.currentSessionId = null;
        // Statistics
        this.stats = {
            totalSent: 0,
            successCount: 0,
            errorCount: 0,
            totalResponseTime: 0,
            avgResponseTime: 0,
            queueSize: 0,
            isInitialized: false
        };
        // Default configuration
        this.config = {
            apiEndpoint: config.apiEndpoint || '/api/v1/events',
            timestampEndpoint: config.timestampEndpoint || '/api/v1/timestamp',
            encryptionMode: 'plaintext', // Fixed to plaintext mode
            appId: (_a = config.appId) !== null && _a !== void 0 ? _a : null,
            bundle: config.bundle || 'unknown-app',
            language: config.language || 'zh-CN',
            network: config.network || 'unknown',
            appVersion: config.appVersion || '1.0.0',
            osVersion: config.osVersion || '',
            enableAutoTimestamp: config.enableAutoTimestamp !== false,
            enableRetry: config.enableRetry !== false,
            maxRetries: config.maxRetries || 3,
            retryDelay: config.retryDelay || 1000,
            batchSize: config.batchSize || 10,
            batchInterval: config.batchInterval || 5000,
            enableBatching: config.enableBatching || false,
            debug: config.debug || false
        };
        this.log(`Cocos Event Tracker SDK v${CocosEventTracker.SDK_VERSION} initialized`, this.config);
    }
    /**
     * Initialize the SDK with server time synchronization
     */
    async init(userId = null, sessionId = null) {
        try {
            this.log(`Initializing Cocos Event Tracker SDK v${CocosEventTracker.SDK_VERSION}...`);
            this.currentUserId = userId;
            this.currentSessionId = sessionId;
            // Get server timestamp for time synchronization
            if (this.config.enableAutoTimestamp) {
                await this.syncServerTime();
            }
            // Start batch processing if enabled
            if (this.config.enableBatching) {
                this.startBatchProcessor();
            }
            this.isInitialized = true;
            this.stats.isInitialized = true;
            this.log(`Cocos Event Tracker SDK v${CocosEventTracker.SDK_VERSION} initialization completed successfully`);
            return true;
        }
        catch (error) {
            this.logError('SDK initialization failed:', error);
            return false;
        }
    }
    /**
     * Synchronize server timestamp
     */
    async syncServerTime() {
        try {
            this.log('Syncing server time...');
            const timestampUrl = this.buildAbsoluteUrl(this.config.timestampEndpoint);
            const response = await fetch(timestampUrl, {
                method: 'GET',
                headers: {
                    'User-Agent': `CocosEventTracker/${CocosEventTracker.SDK_VERSION}`
                }
            });
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            const data = await response.json();
            this.startupServerTimestamp = data.timestamp;
            this.clientInitTime = new Date().toISOString();
            this.log('Server time synchronized:', this.startupServerTimestamp);
            this.log('Client init time recorded:', this.clientInitTime);
            return this.startupServerTimestamp;
        }
        catch (error) {
            this.logError('Failed to sync server time:', error);
            // Use local time as fallback
            this.startupServerTimestamp = new Date().toISOString();
            this.log('Using local time as fallback:', this.startupServerTimestamp);
            return this.startupServerTimestamp;
        }
    }
    /**
     * Set current user information
     */
    setUser(userId, sessionId = null) {
        this.currentUserId = userId;
        if (sessionId) {
            this.currentSessionId = sessionId;
        }
        this.log('User set:', { userId, sessionId: this.currentSessionId });
    }
    /**
     * Track a single event
     */
    async track(eventType, fields = {}, eventTimestamp = null) {
        if (!this.isInitialized) {
            this.logError('SDK not initialized. Call init() first.');
            return false;
        }
        if (!eventType) {
            this.logError('Event type is required');
            return false;
        }
        const event = {
            event_type: eventType,
            event_timestamp: eventTimestamp || new Date().toISOString(),
            fields: fields
        };
        if (this.config.enableBatching) {
            this.addToQueue(event);
            return true;
        }
        else {
            return await this.sendEvents([event]);
        }
    }
    /**
     * Track multiple events at once
     */
    async trackBatch(events) {
        if (!this.isInitialized) {
            this.logError('SDK not initialized. Call init() first.');
            return false;
        }
        if (!Array.isArray(events) || events.length === 0) {
            this.logError('Events must be a non-empty array');
            return false;
        }
        const processedEvents = events.map(event => ({
            event_type: event.eventType,
            event_timestamp: event.eventTimestamp || new Date().toISOString(),
            fields: event.fields || {}
        }));
        if (this.config.enableBatching) {
            processedEvents.forEach(event => this.addToQueue(event));
            return true;
        }
        else {
            return await this.sendEvents(processedEvents);
        }
    }
    /**
     * Add event to queue for batch processing
     */
    addToQueue(event) {
        this.eventQueue.push(event);
        this.log('Event added to queue:', event.event_type);
        this.updateQueueSize();
        if (this.eventQueue.length >= this.config.batchSize) {
            this.flushQueue();
        }
    }
    /**
     * Flush event queue
     */
    async flushQueue() {
        if (this.eventQueue.length === 0) {
            return true;
        }
        const events = [...this.eventQueue];
        this.eventQueue = [];
        this.updateQueueSize();
        this.log(`Flushing ${events.length} events from queue`);
        return await this.sendEvents(events);
    }
    /**
     * Start batch processor
     */
    startBatchProcessor() {
        if (this.batchTimer) {
            clearInterval(this.batchTimer);
        }
        this.batchTimer = setInterval(() => {
            if (this.eventQueue.length > 0) {
                this.flushQueue();
            }
        }, this.config.batchInterval);
        this.log('Batch processor started with interval:', this.config.batchInterval);
    }
    /**
     * Stop batch processor
     */
    stopBatchProcessor() {
        if (this.batchTimer) {
            clearInterval(this.batchTimer);
            this.batchTimer = null;
            this.log('Batch processor stopped');
        }
    }
    /**
     * Send events to server
     */
    async sendEvents(events) {
        try {
            // Build event data structure
            const eventData = this.buildEventData(events);
            // Use POST method with Base64 encoded data
            let url, requestOptions;
            const base64Data = btoa(unescape(encodeURIComponent(JSON.stringify(eventData))));
            url = this.buildAbsoluteUrl(this.config.apiEndpoint);
            requestOptions = {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                    'User-Agent': `CocosEventTracker/${CocosEventTracker.SDK_VERSION}`
                },
                body: `data=${base64Data}`
            };
            this.log('Sending events:', {
                count: events.length,
                url: url
            });
            const startTime = Date.now();
            const response = await fetch(url, requestOptions);
            const responseTime = Date.now() - startTime;
            const responseText = await response.text();
            if (response.ok) {
                this.stats.totalSent++;
                this.stats.successCount++;
                this.stats.totalResponseTime += responseTime;
                this.updateAvgResponseTime();
                this.log('Events sent successfully:', {
                    status: response.status,
                    responseTime: `${responseTime}ms`,
                    response: responseText
                });
                return true;
            }
            else {
                this.stats.totalSent++;
                this.stats.errorCount++;
                this.logError('Failed to send events:', {
                    status: response.status,
                    responseTime: `${responseTime}ms`,
                    response: responseText
                });
                // Retry logic
                if (this.config.enableRetry && this.stats.errorCount <= this.config.maxRetries) {
                    this.log(`Retrying in ${this.config.retryDelay}ms...`);
                    setTimeout(() => this.sendEvents(events), this.config.retryDelay);
                }
                return false;
            }
        }
        catch (error) {
            this.stats.errorCount++;
            this.logError('Error sending events:', error);
            return false;
        }
    }
    /**
     * Build event data structure
     */
    buildEventData(events) {
        return {
            version: CocosEventTracker.DATA_FORMAT_VERSION,
            server_init_time: this.startupServerTimestamp,
            user_id: this.currentUserId || null,
            session_id: this.currentSessionId,
            client_time: new Date().toISOString(),
            client_init_time: this.clientInitTime,
            app_id: this.config.appId,
            language: this.config.language,
            network: this.config.network,
            app_version: this.config.appVersion,
            os_version: this.config.osVersion,
            bundle: this.config.bundle,
            events: events
        };
    }
    /**
     * Build absolute URL
     */
    buildAbsoluteUrl(endpoint) {
        if (endpoint.startsWith('http://') || endpoint.startsWith('https://')) {
            return endpoint;
        }
        // Handle relative URLs (browser environment)
        if (typeof window !== 'undefined') {
            const baseUrl = window.location.origin;
            return endpoint.startsWith('/') ? baseUrl + endpoint : baseUrl + '/' + endpoint;
        }
        // Node.js environment - return as-is
        return endpoint;
    }
    /**
     * Get SDK statistics
     */
    getStats() {
        return { ...this.stats };
    }
    /**
     * Reset statistics
     */
    resetStats() {
        this.stats = {
            totalSent: 0,
            successCount: 0,
            errorCount: 0,
            totalResponseTime: 0,
            avgResponseTime: 0,
            queueSize: this.eventQueue.length,
            isInitialized: this.isInitialized
        };
        this.log('Statistics reset');
    }
    /**
     * Destroy SDK and cleanup resources
     */
    destroy() {
        this.stopBatchProcessor();
        // Flush remaining events
        if (this.eventQueue.length > 0) {
            this.flushQueue();
        }
        this.isInitialized = false;
        this.stats.isInitialized = false;
        this.log(`Cocos Event Tracker SDK v${CocosEventTracker.SDK_VERSION} destroyed`);
    }
    /**
     * Logging utility
     */
    log(...args) {
        if (this.config.debug) {
            console.log('[CocosEventTracker]', ...args);
        }
    }
    /**
     * Error logging utility
     */
    logError(...args) {
        if (this.config.debug) {
            console.error('[CocosEventTracker]', ...args);
        }
    }
    /**
     * Update queue size in stats
     */
    updateQueueSize() {
        this.stats.queueSize = this.eventQueue.length;
    }
    /**
     * Update average response time
     */
    updateAvgResponseTime() {
        if (this.stats.successCount > 0) {
            this.stats.avgResponseTime = Math.round(this.stats.totalResponseTime / this.stats.successCount);
        }
    }
}
CocosEventTracker.SDK_VERSION = '1.0.4';
CocosEventTracker.DATA_FORMAT_VERSION = '1.0';

export { CocosEventTracker, CocosEventTracker as default };//# sourceMappingURL=https://main.vscode-cdn.net/sourcemaps/d90c647ab7bea1f6f5db03260a313631e1e8d27c/node_modules/@ccserv/event-tracker/dist/index.esm.js.map
