"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.onMcpServerStarted = void 0;
exports.activate = activate;
exports.deactivate = deactivate;
const vscode = __importStar(require("vscode"));
const fs = __importStar(require("fs"));
const path = __importStar(require("path"));
const mcpServer_1 = require("./mcpServer");
/**
 * MCP 服务器定义提供者的唯一 ID，用于向 VS Code 注册并识别该提供者
 */
const PROVIDER_ID = 'cocos-cli-mcp-provider';
/**
 * MCP 服务器启动成功事件发射器（供外部订阅）
 */
let _onMcpServerStarted;
/**
 * 弹出带“Open Settings”按钮的消息，并在点击后跳转到 `cocos.enginePath` 设置
 * @param severity 消息级别：warning 或 error
 * @param message 显示的消息内容
 */
async function promptOpenSettings(severity, message) {
    const label = vscode.l10n.t('Install SDK');
    const pick = severity === 'warning'
        ? await vscode.window.showWarningMessage(message, label)
        : await vscode.window.showErrorMessage(message, label);
    if (pick === label) {
        vscode.commands.executeCommand('workbench.view.cocos.packageView.focus');
    }
}
/**
 * 扩展激活入口：注册 MCP 服务器定义提供者，并监听工作区与设置变化
 */
async function activate(context) {
    // 创建事件发射器，用于通知 MCP 服务器定义变化
    const onDidChangeMcpServerDefinitionsEmitter = new vscode.EventEmitter();
    /** 是否已启动过 MCP 服务器（避免重复启动） */
    let hasStartedMcpServer = false;
    /** 当前已启动的引擎路径（与缓存匹配时可复用定义） */
    let activeEnginePath;
    /** 已缓存的 MCP 服务器定义（提高后续查询效率） */
    let cachedDefinitions;
    /** 最近一次有效的引擎路径（用于提示与变更） */
    let lastKnownEnginePath = getConfiguredEnginePath();
    /** 启动过程中显示的状态栏项（单例，避免重复显示） */
    let startupStatus;
    /** 启动状态标记（控制状态栏显示与隐藏） */
    let isStarting = false;
    /** 服务器启动中的并发锁，避免重复启动 */
    let startupPromise;
    /** MCP 服务器管理器 */
    const mcpManager = new mcpServer_1.McpServerManager();
    context.subscriptions.push(mcpManager);
    // 监听 MCP 服务器停止事件，重置状态并通知 workbench
    context.subscriptions.push(mcpManager.onDidStop(() => {
        hasStartedMcpServer = false;
        activeEnginePath = undefined;
        cachedDefinitions = undefined;
        // 通知 workbench MCP 服务器状态已变化
        vscode.commands.executeCommand('cocos.internal.onMcpServerStatusChanged');
    }));
    // 初始化 MCP 服务器启动成功事件发射器
    _onMcpServerStarted = new vscode.EventEmitter();
    exports.onMcpServerStarted = _onMcpServerStarted.event;
    context.subscriptions.push(_onMcpServerStarted);
    /** MCP 服务器定义提供者 */
    const provider = {
        onDidChangeMcpServerDefinitions: onDidChangeMcpServerDefinitionsEmitter.event,
        /**
         * 提供 MCP 服务器定义：校验项目与引擎环境，启动服务并返回定义
         */
        provideMcpServerDefinitions: async (token) => {
            let folder = getCurrentProjectFolder();
            if (!folder) {
                vscode.window.showWarningMessage(vscode.l10n.t('No Cocos project is open.'));
                return [];
            }
            // 检查是否为 Cocos 工程
            const checkResult = await checkIsCocosProject(folder);
            if (!checkResult.res) {
                if (checkResult.message) {
                    vscode.window.showErrorMessage(checkResult.message);
                }
                return []; // 不启动 MCP 服务器，也不返回任何定义
            }
            try {
                const enginePath = getConfiguredEnginePath();
                if (!enginePath) {
                    await promptOpenSettings('warning', vscode.l10n.t('Configure the Cocos engine path in settings first.'));
                    return [];
                }
                if (!fs.existsSync(enginePath)) {
                    await promptOpenSettings('error', vscode.l10n.t('The Cocos engine path does not exist: {0}', enginePath));
                    return [];
                }
                if (hasStartedMcpServer && enginePath === activeEnginePath && cachedDefinitions) {
                    return cachedDefinitions;
                }
                // 启动 MCP 服务器（并发锁控制）
                if (startupPromise) {
                    await startupPromise;
                    return cachedDefinitions ?? [];
                }
                if (!startupStatus) {
                    startupStatus = vscode.window.createStatusBarItem(vscode.StatusBarAlignment.Left);
                    context.subscriptions.push(startupStatus);
                }
                startupStatus.text = `$(loading) ${vscode.l10n.t('Starting Cocos MCP Server…')}`;
                startupStatus.tooltip = vscode.l10n.t('Starting Cocos MCP Server');
                startupStatus.show();
                isStarting = true;
                startupPromise = (async () => {
                    const url = await mcpManager.start(folder, enginePath);
                    if (!url) {
                        return;
                    }
                    const definitions = [
                        new vscode.McpHttpServerDefinition(vscode.l10n.t('Cocos CLI MCP Server'), vscode.Uri.parse(url))
                    ];
                    hasStartedMcpServer = true;
                    activeEnginePath = enginePath;
                    cachedDefinitions = definitions;
                    lastKnownEnginePath = enginePath;
                    // 广播 MCP 服务器启动成功事件
                    _onMcpServerStarted.fire({
                        url,
                        enginePath,
                        projectPath: folder
                    });
                })();
                await startupPromise;
                startupPromise = undefined;
                return cachedDefinitions ?? [];
            }
            catch (error) {
                console.error(vscode.l10n.t('Failed to start the MCP server.'), error);
                return [];
            }
            finally {
                console.log('======= MCP server started successfully.======');
                if (isStarting) {
                    startupStatus?.hide();
                    isStarting = false;
                }
            }
        },
        /**
         * 解析 MCP 服务器定义：此处直接透传，可在需要时加入额外校验或交互
         */
        resolveMcpServerDefinition: async (definition, token) => {
            // 可以在这里做额外检查 / 用户交互 / 获取 token 等
            // 如果一切正常，直接返回 definition 即可
            return definition;
        }
    };
    // 注册 MCP 服务器定义提供者
    const disposable = vscode.lm.registerMcpServerDefinitionProvider(PROVIDER_ID, provider);
    context.subscriptions.push(disposable);
    // 监听工作区变化，当工作区变化时通知 MCP 服务器定义可能发生变化
    context.subscriptions.push(vscode.workspace.onDidChangeWorkspaceFolders(() => {
        onDidChangeMcpServerDefinitionsEmitter.fire();
    }));
    // 启动时如果配置已完成且路径可用，则立即尝试启动
    void (async () => {
        const folder = getCurrentProjectFolder();
        if (!folder) {
            return;
        }
        if (!(await checkIsCocosProject(folder))) {
            return;
        }
        const enginePath = getConfiguredEnginePath();
        if (!enginePath || !fs.existsSync(enginePath)) {
            return;
        }
        onDidChangeMcpServerDefinitionsEmitter.fire();
    })();
    // 注册检查 Cocos 项目的命令，供其他插件（如 Chat）调用
    context.subscriptions.push(vscode.commands.registerCommand('cocos.checkIsCocosProject', async () => {
        const folder = getCurrentProjectFolder();
        if (!folder) {
            return { res: false, message: vscode.l10n.t('No project is open.') };
        }
        return checkIsCocosProject(folder);
    }));
    // 注册获取 MCP 服务器状态的命令，供其他插件查询
    context.subscriptions.push(vscode.commands.registerCommand('cocos.getMcpServerStatus', () => {
        const definition = cachedDefinitions?.[0];
        let url;
        if (definition && 'url' in definition) {
            url = definition.url?.toString();
        }
        return {
            isStarted: hasStartedMcpServer,
            url,
            enginePath: activeEnginePath
        };
    }));
    // 订阅 MCP 服务器启动成功事件，通知 workbench 状态变化
    context.subscriptions.push((0, exports.onMcpServerStarted)(() => {
        // 通知 workbench MCP 服务器状态已变化
        vscode.commands.executeCommand('cocos.internal.onMcpServerStatusChanged');
    }));
    // 监听设置变化，关注 `cocos.enginePath` 变更并提示或刷新定义
    const configListener = vscode.workspace.onDidChangeConfiguration(async (event) => {
        if (event.affectsConfiguration('cocos.enginePath')) {
            const newPath = getConfiguredEnginePath();
            if (!newPath) {
                if (lastKnownEnginePath) {
                    vscode.window.showWarningMessage(vscode.l10n.t('The Cocos engine path has been cleared. Please configure it again.'));
                }
                lastKnownEnginePath = undefined;
                return;
            }
            if (!fs.existsSync(newPath)) {
                const openSettings = vscode.l10n.t('Open Settings');
                const pick = await vscode.window.showErrorMessage(vscode.l10n.t('The Cocos engine path does not exist: {0}', newPath), openSettings);
                if (pick === openSettings) {
                    vscode.commands.executeCommand('workbench.action.openSettings', 'cocos.enginePath');
                }
                return;
            }
            lastKnownEnginePath = newPath;
            if (hasStartedMcpServer) {
                if (newPath !== activeEnginePath) {
                    vscode.window.showInformationMessage(vscode.l10n.t('The Cocos engine path has been updated and will take effect after the next start.'));
                }
                return;
            }
            onDidChangeMcpServerDefinitionsEmitter.fire();
        }
    });
    context.subscriptions.push(configListener);
    // 返回 API 供其他插件使用
    return {
        onMcpServerStarted: exports.onMcpServerStarted
    };
}
/**
 * 扩展停用钩子：当前无需清理，VS Code 会回收已注册的资源
 */
function deactivate() { }
/**
 * 检查当前文件夹是否为 Cocos 工程
 * 依据：存在 `package.json` 且包含 `creator` 字段
 * @param folderPath 文件夹路径
 * @returns 检查结果对象
 */
async function checkIsCocosProject(folderPath) {
    try {
        const packageJsonPath = path.join(folderPath, 'package.json');
        // 检查 package.json 是否存在
        if (!fs.existsSync(packageJsonPath)) {
            return { res: false, message: vscode.l10n.t('The current workspace is not a Cocos project.') };
        }
        // 读取并解析 package.json
        const packageJsonContent = fs.readFileSync(packageJsonPath, 'utf8');
        const packageJson = JSON.parse(packageJsonContent);
        // 检查是否有 creator 字段
        if (!packageJson.creator) {
            return { res: false, message: vscode.l10n.t('The current workspace is not a Cocos project.') };
        }
        return { res: true };
    }
    catch (error) {
        return { res: false, message: vscode.l10n.t('The current workspace is not a Cocos project.') };
    }
}
/**
 * 获取当前打开的项目文件夹路径
 * @returns 项目文件夹路径
 */
/**
 * 获取当前打开的项目文件夹路径
 * 优先当前活动文件所在工作区，否则返回第一个工作区
 */
function getCurrentProjectFolder() {
    // 获取当前工作区的第一个文件夹（项目根目录）
    const workspaceFolders = vscode.workspace.workspaceFolders;
    if (!workspaceFolders || workspaceFolders.length === 0) {
        return undefined; // 没有打开任何工作区
    }
    // 如果有多个工作区文件夹，优先返回当前活动文件所在的工作区
    const editor = vscode.window.activeTextEditor;
    if (editor) {
        const currentFileUri = editor.document.uri;
        const workspaceFolder = vscode.workspace.getWorkspaceFolder(currentFileUri);
        if (workspaceFolder) {
            return workspaceFolder.uri.fsPath;
        }
    }
    // 否则返回第一个工作区文件夹
    return workspaceFolders[0].uri.fsPath;
}
/**
 * 获取配置中的 Cocos 引擎路径并去除空白
 * @returns 规范化后的路径（若为空则返回 undefined）
 */
function getConfiguredEnginePath() {
    const rawPath = vscode.workspace.getConfiguration('cocos').get('enginePath');
    const normalized = rawPath?.trim();
    return normalized ? normalized : undefined;
}
//# sourceMappingURL=extension.js.map